# -*- coding: utf-8 -*-
'''
***********************************************************
*
* @file addon.py
* @package plugin.video.shteryanovlive
*
* ShteryanovLive - Live TV and Sports Streaming
* A reborn version with improved backend integration
*
* Created on 2026-02-05
* Copyright 2026 by Shteryanov. All rights reserved.
*
* @license GNU General Public License, version 3 (GPL-3.0)
*
***********************************************************
'''

import re
import os
import sys
import json
import html
import base64

from urllib.parse import urlencode, unquote, parse_qsl, quote_plus, urlparse, urljoin
from datetime import datetime, timezone

import requests
import xbmc
import xbmcvfs
import xbmcgui
import xbmcplugin
import xbmcaddon


# =============================================================================
# Addon Configuration
# =============================================================================

ADDON_ID = 'plugin.video.shteryanovlive'
ADDON_NAME = 'ShteryanovLive'

addon_url = sys.argv[0]
addon_handle = int(sys.argv[1])
params = dict(parse_qsl(sys.argv[2][1:]))
addon = xbmcaddon.Addon(id=ADDON_ID)
mode = addon.getSetting('mode')

UA = 'Mozilla/5.0 (Macintosh; Intel Mac OS X 10_15_7) AppleWebKit/537.36 (KHTML, like Gecko) Chrome/136.0.0.0 Safari/537.36'
FANART = addon.getAddonInfo('fanart')
ICON = addon.getAddonInfo('icon')

# DaddyLive source URL (for scraping schedules/channels)
SEED_BASEURL = 'https://dlhd.link/'

# Backend API URL (our custom backend for CDN auth)
BACKEND_URL = addon.getSetting('backend_url') or 'http://localhost:8080'

# CDN Configuration (discovered from investigation)
CDN_DOMAINS = {
    'primary': 'dvalna.ru',
    'server_lookup': 'chevy.dvalna.ru',
}
PLAYER_ORIGIN = 'https://codepcplay.fun'


# =============================================================================
# Logging
# =============================================================================

def log(msg):
    logpath = xbmcvfs.translatePath('special://logpath/')
    filename = 'shteryanovlive.log'
    log_file = os.path.join(logpath, filename)
    try:
        if isinstance(msg, str):
            _msg = f'\n    {msg}'
        else:
            _msg = f'\n    {repr(msg)}'
        if not os.path.exists(log_file):
            with open(log_file, 'w', encoding='utf-8'):
                pass
        with open(log_file, 'a', encoding='utf-8') as f:
            line = f'[{datetime.now().date()} {str(datetime.now().time())[:8]}]: {_msg}'
            f.write(line.rstrip('\r\n') + '\n')
    except (TypeError, Exception) as e:
        try:
            xbmc.log(f'[ {ADDON_NAME} ] Logging Failure: {e}', 2)
        except:
            pass


# =============================================================================
# URL Helpers
# =============================================================================

def normalize_origin(url):
    try:
        u = urlparse(url)
        return f'{u.scheme}://{u.netloc}/'
    except:
        return SEED_BASEURL


def resolve_active_baseurl(seed):
    try:
        s = requests.Session()
        headers = {'User-Agent': UA}
        resp = s.get(seed, headers=headers, timeout=10, allow_redirects=True)
        final = normalize_origin(resp.url if resp.url else seed)
        return final
    except Exception as e:
        log(f'Active base resolve failed, using seed. Error: {e}')
        return normalize_origin(seed)


def get_active_base():
    base = addon.getSetting('active_baseurl')
    if not base:
        base = resolve_active_baseurl(SEED_BASEURL)
        addon.setSetting('active_baseurl', base)
    if not base.endswith('/'):
        base += '/'
    return base


def set_active_base(new_base: str):
    if not new_base.endswith('/'):
        new_base += '/'
    addon.setSetting('active_baseurl', new_base)


def abs_url(path: str) -> str:
    return urljoin(get_active_base(), path.lstrip('/'))


def _sched_headers():
    base = get_active_base()
    return {'User-Agent': UA, 'Referer': base, 'Origin': base}


# =============================================================================
# Time Conversion
# =============================================================================

def get_local_time(utc_time_str):
    try:
        utc_now = datetime.utcnow()
        event_time_utc = datetime.strptime(utc_time_str, '%H:%M')
        event_time_utc = event_time_utc.replace(year=utc_now.year, month=utc_now.month, day=utc_now.day)
        event_time_utc = event_time_utc.replace(tzinfo=timezone.utc)
        local_time = event_time_utc.astimezone()
        time_format_pref = addon.getSetting('time_format')
        if time_format_pref == '1':
            local_time_str = local_time.strftime('%H:%M')
        else:
            local_time_str = local_time.strftime('%I:%M %p').lstrip('0')
        return local_time_str
    except Exception as e:
        log(f"Failed to convert time: {e}")
        return utc_time_str


# =============================================================================
# Kodi UI Helpers
# =============================================================================

def build_url(query):
    return addon_url + '?' + urlencode(query)


def addDir(title, dir_url, is_folder=True):
    li = xbmcgui.ListItem(title)
    clean_plot = re.sub(r'<[^>]+>', '', title)
    labels = {'title': title, 'plot': clean_plot, 'mediatype': 'video'}
    kodiversion = getKodiversion()
    if kodiversion < 20:
        li.setInfo("video", labels)
    else:
        infotag = li.getVideoInfoTag()
        infotag.setMediaType(labels.get("mediatype", "video"))
        infotag.setTitle(labels.get("title", ADDON_NAME))
        infotag.setPlot(labels.get("plot", labels.get("title", ADDON_NAME)))
    li.setArt({'thumb': '', 'poster': '', 'banner': '', 'icon': ICON, 'fanart': FANART})
    li.setProperty("IsPlayable", 'false' if is_folder else 'true')
    xbmcplugin.addDirectoryItem(handle=addon_handle, url=dir_url, listitem=li, isFolder=is_folder)


def closeDir():
    xbmcplugin.endOfDirectory(addon_handle)


def getKodiversion():
    try:
        return int(xbmc.getInfoLabel("System.BuildVersion")[:2])
    except:
        return 18


# =============================================================================
# Main Menu
# =============================================================================

def Main_Menu():
    menu = [
        [f'[B][COLOR gold]LIVE SPORTS SCHEDULE[/COLOR][/B]', 'sched'],
        [f'[B][COLOR gold]LIVE TV CHANNELS[/COLOR][/B]', 'live_tv'],
        [f'[B][COLOR gold]SEARCH EVENTS SCHEDULE[/COLOR][/B]', 'search'],
        [f'[B][COLOR gold]SEARCH LIVE TV CHANNELS[/COLOR][/B]', 'search_channels'],
        [f'[B][COLOR gold]REFRESH CATEGORIES[/COLOR][/B]', 'refresh_sched'],
        [f'[B][COLOR gold]SET ACTIVE DOMAIN (AUTO)[/COLOR][/B]', 'resolve_base_now'],
        [f'[B][COLOR cyan]BACKEND STATUS[/COLOR][/B]', 'backend_status'],
    ]
    for m in menu:
        addDir(m[0], build_url({'mode': 'menu', 'serv_type': m[1]}))
    closeDir()


# =============================================================================
# Backend API Integration
# =============================================================================

def get_backend_url():
    """Get the backend URL from settings or use default."""
    url = addon.getSetting('backend_url')
    if not url:
        url = 'http://localhost:8080'
    return url.rstrip('/')


def check_backend_status():
    """Check if the backend is accessible."""
    try:
        backend = get_backend_url()
        resp = requests.get(f'{backend}/', timeout=5)
        if resp.status_code == 200:
            data = resp.json()
            return True, data
        return False, {'error': f'HTTP {resp.status_code}'}
    except Exception as e:
        return False, {'error': str(e)}


def get_auth_token_from_backend(channel_id):
    """Fetch authentication token from our backend API."""
    try:
        backend = get_backend_url()
        url = f'{backend}/api/auth_token/{channel_id}'
        log(f'[Backend] Fetching auth token from: {url}')
        
        resp = requests.get(url, timeout=15)
        if resp.status_code == 200:
            data = resp.json()
            log(f'[Backend] Got auth token for channel {channel_id}')
            return data
        else:
            log(f'[Backend] Auth token request failed: {resp.status_code}')
            return None
    except Exception as e:
        log(f'[Backend] Auth token error: {e}')
        return None


def get_stream_url_from_backend(channel_id):
    """Get complete stream URL from our backend API."""
    try:
        backend = get_backend_url()
        url = f'{backend}/api/stream_url/{channel_id}'
        log(f'[Backend] Fetching stream URL from: {url}')
        
        resp = requests.get(url, timeout=15)
        if resp.status_code == 200:
            data = resp.json()
            log(f'[Backend] Got stream URL: {data.get("hls_url")}')
            return data
        else:
            log(f'[Backend] Stream URL request failed: {resp.status_code}')
            return None
    except Exception as e:
        log(f'[Backend] Stream URL error: {e}')
        return None


def get_proxy_stream_from_backend(channel_id):
    """
    Get a fully proxied stream URL from our backend API.
    
    This returns a proxy URL that doesn't require any authentication
    headers from the client - the backend handles all auth internally.
    """
    try:
        backend = get_backend_url()
        url = f'{backend}/api/proxy_stream/{channel_id}'
        log(f'[Backend] Fetching proxy stream URL from: {url}')
        
        resp = requests.get(url, timeout=15)
        if resp.status_code == 200:
            data = resp.json()
            proxy_url = data.get('proxy_url')
            log(f'[Backend] Got proxy URL: {proxy_url}')
            return data
        else:
            log(f'[Backend] Proxy stream request failed: {resp.status_code}')
            return None
    except Exception as e:
        log(f'[Backend] Proxy stream error: {e}')
        return None


def show_backend_status():
    """Show backend status in a dialog."""
    status, data = check_backend_status()
    if status:
        msg = (
            f"Backend: ONLINE\n"
            f"Service: {data.get('service', 'unknown')}\n"
            f"Version: {data.get('version', 'unknown')}\n"
            f"CDN: {data.get('cdn_domain', 'unknown')}"
        )
    else:
        msg = f"Backend: OFFLINE\n\nError: {data.get('error', 'Unknown')}\n\nURL: {get_backend_url()}"
    
    xbmcgui.Dialog().ok(ADDON_NAME, msg)


# =============================================================================
# Category/Schedule Parsing
# =============================================================================

def getCategTrans():
    schedule_url = abs_url('index.php')
    try:
        headers = {'User-Agent': UA, 'Referer': get_active_base()}
        html_text = requests.get(schedule_url, headers=headers, timeout=10).text

        m = re.search(r'<div[^>]+class="filters"[^>]*>(.*?)</div>', html_text, re.IGNORECASE | re.DOTALL)
        if not m:
            log("getCategTrans(): filters block not found")
            return []

        block = m.group(1)
        anchors = re.findall(r'<a[^>]+href="([^"]+)"[^>]*>(.*?)</a>', block, re.IGNORECASE | re.DOTALL)
        if not anchors:
            log("getCategTrans(): no <a> items in filters block")
            return []

        categs = []
        seen = set()

        for href, text_content in anchors:
            name = html.unescape(re.sub(r'\s+', ' ', text_content)).strip()
            if not name or name.lower() == 'all':
                continue
            if name in seen:
                continue
            seen.add(name)
            categs.append((name, '[]'))

        return categs

    except Exception as e:
        xbmcgui.Dialog().ok("Error", f"Error fetching category data: {e}")
        log(f'index parse fail: url={schedule_url} err={e}')
        return []


def Menu_Trans():
    categs = getCategTrans()
    if not categs:
        return
    for categ_name, _events_list in categs:
        addDir(categ_name, build_url({'mode': 'showChannels', 'trType': categ_name}))
    closeDir()


def ShowChannels(categ, channels_list):
    for item in channels_list:
        title = item.get('title')
        addDir(title, build_url({'mode': 'trList', 'trType': categ, 'channels': json.dumps(item.get('channels'))}), True)
    closeDir()


def getTransData(categ):
    try:
        url = abs_url('index.php?cat=' + quote_plus(categ))
        headers = {'User-Agent': UA, 'Referer': get_active_base()}
        html_text = requests.get(url, headers=headers, timeout=10).text

        cut = re.search(r'<h2\s+class="collapsible-header\b', html_text, re.IGNORECASE)
        if cut:
            html_text = html_text[:cut.start()]

        events = re.findall(
            r'<div\s+class="schedule__event">.*?'
            r'<div\s+class="schedule__eventHeader"[^>]*?>\s*'
            r'(?:<[^>]+>)*?'
            r'<span\s+class="schedule__time"[^>]*data-time="([^"]+)"[^>]*>.*?</span>\s*'
            r'<span\s+class="schedule__eventTitle">\s*([^<]+)\s*</span>.*?'
            r'</div>\s*'
            r'<div\s+class="schedule__channels">(.*?)</div>',
            html_text, re.IGNORECASE | re.DOTALL
        )

        trns = []
        for time_str, event_title, channels_block in events:
            event_time_local = get_local_time(time_str.strip())
            title = f'[COLOR gold]{event_time_local}[/COLOR] {html.unescape(event_title.strip())}'

            chans = []
            for href, title_attr, link_text in re.findall(
                r'<a[^>]+href="([^"]+)"[^>]*title="([^"]*)"[^>]*>(.*?)</a>',
                channels_block, re.IGNORECASE | re.DOTALL
            ):
                try:
                    u = urlparse(href)
                    qs = dict(parse_qsl(u.query))
                    cid = qs.get('id') or ''
                except Exception:
                    cid = ''
                name = html.unescape((title_attr or link_text).strip())
                if cid:
                    chans.append({'channel_name': name, 'channel_id': cid})

            if chans:
                trns.append({'title': title, 'channels': chans})

        return trns

    except Exception as e:
        log(f'getTransData error for categ={categ}: {e}')
        return []


def TransList(categ, channels):
    for channel in channels:
        channel_title = html.unescape(channel.get('channel_name'))
        channel_id = str(channel.get('channel_id', '')).strip()
        if not channel_id:
            continue
        addDir(channel_title, build_url({'mode': 'trLinks', 'trData': json.dumps({'channels': [{'channel_name': channel_title, 'channel_id': channel_id}]})}), False)
    closeDir()


def getSource(trData):
    try:
        data = json.loads(unquote(trData))
        channels_data = data.get('channels')
        if channels_data and isinstance(channels_data, list):
            cid = str(channels_data[0].get('channel_id', '')).strip()
            if not cid:
                return
            if '%7C' in cid or '|' in cid:
                url_stream = abs_url('watchs2watch.php?id=' + cid)
            else:
                url_stream = abs_url('watch.php?id=' + cid)
            xbmcplugin.setContent(addon_handle, 'videos')
            PlayStream(url_stream)
    except Exception as e:
        log(f'getSource failed: {e}')


# =============================================================================
# Live TV Channel List
# =============================================================================

def list_gen():
    chData = channels()
    for c in chData:
        addDir(c[1], build_url({'mode': 'play', 'url': abs_url(c[0])}), False)
    closeDir()


def channels():
    url = abs_url('24-7-channels.php')
    allow_adult = xbmcaddon.Addon().getSetting('adult_pw') == 'lol'
    headers = {'Referer': get_active_base(), 'User-Agent': UA}

    try:
        resp = requests.post(url, headers=headers, timeout=10).text
    except Exception as e:
        log(f"[{ADDON_NAME}] channels(): request failed: {e}")
        return []

    card_rx = re.compile(
        r'<a\s+class="card"[^>]*?href="(?P<href>[^"]+)"[^>]*?data-title="(?P<data_title>[^"]*)"[^>]*>'
        r'.*?<div\s+class="card__title">\s*(?P<title>.*?)\s*</div>'
        r'.*?ID:\s*(?P<id>\d+)\s*</div>'
        r'.*?</a>',
        re.IGNORECASE | re.DOTALL
    )

    items = []
    for m in card_rx.finditer(resp):
        href_rel = m.group('href').strip()
        title_dom = html.unescape(m.group('title').strip())
        title_attr = html.unescape(m.group('data_title').strip())
        name = title_dom or title_attr

        is_adult = ('18+' in name) or ('18+' in title_attr) or name.lower().startswith('18+')
        if is_adult and not allow_adult:
            continue

        name = re.sub(r'^\s*\d+(?=[A-Za-z])', '', name).strip()

        items.append([href_rel, name])

    if not items:
        log(f"[{ADDON_NAME}] channels(): Site may have changed.")

    return items


# =============================================================================
# Stream Playback - With Backend Integration
# =============================================================================

def PlayStream(link):
    try:
        # Extract channel ID from URL
        try:
            url_parts = urlparse(link)
            query_params = dict(parse_qsl(url_parts.query))
            channel_id = query_params.get('id', '')
        except:
            channel_id = ''
        
        if not channel_id:
            # Try to extract from path
            m = re.search(r'id[=/-](\d+)', link)
            if m:
                channel_id = m.group(1)
        
        log(f'[PlayStream] Channel ID: {channel_id}')
        
        if not channel_id:
            log('[PlayStream] Could not extract channel ID from URL')
            xbmcgui.Dialog().ok(ADDON_NAME, "Could not determine channel ID")
            return
        
        # Check if backend is available
        use_backend = addon.getSetting('use_backend') != 'false'
        backend_available = False
        
        if use_backend:
            status, _ = check_backend_status()
            backend_available = status
            log(f'[PlayStream] Backend available: {backend_available}')
        
        if backend_available:
            # Use backend API to get fully proxied stream URL
            log('[PlayStream] Using backend proxy API for stream')
            proxy_data = get_proxy_stream_from_backend(channel_id)
            
            if proxy_data:
                proxy_url = proxy_data.get('proxy_url')
                
                if proxy_url:
                    log(f'[PlayStream] Playing proxy URL: {proxy_url}')
                    # Play the proxy URL directly - no headers needed!
                    _play_hls(proxy_url)
                    return
            
            log('[PlayStream] Backend proxy stream failed, falling back to direct method')
        
        # Fallback: Direct stream extraction (original method)
        log('[PlayStream] Using direct stream extraction')
        PlayStreamDirect(link)
        
    except Exception as e:
        import traceback
        log(f"[PlayStream] Exception: {e}\n{traceback.format_exc()}")
        xbmcgui.Dialog().ok(ADDON_NAME, f"Playback error: {str(e)}")


def PlayStreamDirect(link):
    """Original direct stream extraction method (fallback)."""
    try:
        base = get_active_base()
        session = requests.Session()

        def _origin(u):
            try:
                p = urlparse(u)
                return f'{p.scheme}://{p.netloc}'
            except:
                return ''

        def _fetch(u, ref=None, note=''):
            headers = {
                'User-Agent': UA,
                'Referer': base,
                'Origin': _origin(base),
            }
            if ref:
                headers['Referer'] = ref
                headers['Origin'] = _origin(ref)
            log(f'[PlayStreamDirect] GET {u}  (ref={headers.get("Referer")}) {note}')
            r = session.get(u, headers=headers, timeout=10)
            r.raise_for_status()
            return r.text, r.url

        html1, url1 = _fetch(link, ref=base, note='entry')

        m = re.search(r'data-url="([^"]+)"\s+title="PLAYER 2"', html1, re.I)
        if m:
            url2 = urljoin(url1, m.group(1).replace('//cast', '/cast'))
        else:
            m = re.search(r'iframe\s+src="([^"]+)"', html1, re.I)
            if not m:
                log('[PlayStreamDirect] No iframe found after entry page.')
                return
            url2 = urljoin(url1, m.group(1))

        html2, url2_final = _fetch(url2, ref=url1, note='player2/outer iframe')

        m_if = re.search(r'iframe\s+src="([^"]+)"', html2, re.I)
        if m_if:
            url3 = urljoin(url2_final, m_if.group(1))
            html3, url3_final = _fetch(url3, ref=url2_final, note='inner iframe')
            page = html3
            page_url = url3_final
        else:
            page = html2
            page_url = url2_final

        # Extract channel key
        ck_rx = re.compile(
            r'const\s+(?:CHANNEL_KEY|CHANNEL_ID|CH(?:ANNEL)?_?KEY?)\s*=\s*["\']([^"\']+)["\']',
            re.I
        )
        m_ck = ck_rx.search(page)
        
        premium_rx = re.compile(r"'(premium\d+)'", re.I)
        m_premium = premium_rx.search(page)
        
        if m_ck:
            channel_key = m_ck.group(1).strip()
            log(f'[PlayStreamDirect] CHANNEL_KEY from const = {channel_key}')
        elif m_premium:
            channel_key = m_premium.group(1).strip()
            log(f'[PlayStreamDirect] CHANNEL_KEY from premium pattern = {channel_key}')
        else:
            try:
                url_parts = urlparse(link)
                query_params = dict(parse_qsl(url_parts.query))
                cid = query_params.get('id', '')
                if cid:
                    channel_key = f'premium{cid}'
                    log(f'[PlayStreamDirect] CHANNEL_KEY constructed from URL = {channel_key}')
                else:
                    log('[PlayStreamDirect] CHANNEL_KEY not found.')
                    return
            except Exception as e:
                log(f'[PlayStreamDirect] CHANNEL_KEY extraction failed: {e}')
                return

        # Extract auth token from page
        auth_token_match = re.search(r"authToken:\s*['\"]([^'\"]+)['\"]", page)
        auth_token = auth_token_match.group(1) if auth_token_match else None
        
        if auth_token:
            log(f'[PlayStreamDirect] Found auth token in page')
        else:
            log('[PlayStreamDirect] No auth token found in page')

        # Server lookup
        server_key = None
        if auth_token:
            try:
                lookup_url = f'https://{CDN_DOMAINS["server_lookup"]}/server_lookup?channel_id={quote_plus(channel_key)}'
                lookup_headers = {
                    'User-Agent': UA,
                    'Origin': PLAYER_ORIGIN,
                    'Referer': f'{PLAYER_ORIGIN}/',
                    'Cookie': f'eplayer_session={auth_token}',
                }
                lookup_resp = session.get(lookup_url, headers=lookup_headers, timeout=10)
                if lookup_resp.status_code == 200:
                    data = lookup_resp.json()
                    server_key = data.get('server_key', '').strip()
                    log(f'[PlayStreamDirect] server_key from lookup: {server_key}')
            except Exception as e:
                log(f'[PlayStreamDirect] server_lookup failed: {e}')
        
        if not server_key:
            server_key = 'top1'
            log(f'[PlayStreamDirect] Using default server_key: {server_key}')

        # Build HLS URL
        hls_url = f'https://{server_key}new.{CDN_DOMAINS["primary"]}/{server_key}/{channel_key}/mono.css'

        # Build headers
        header_dict = {
            'Referer': f'{PLAYER_ORIGIN}/',
            'Origin': PLAYER_ORIGIN,
            'User-Agent': UA,
        }
        if auth_token:
            header_dict['Cookie'] = f'eplayer_session={auth_token}'
        
        header_str = '&'.join(f'{k}={quote_plus(v)}' for k, v in header_dict.items())
        m3u8 = f'{hls_url}|{header_str}'
        
        log(f'[PlayStreamDirect] Final HLS = {m3u8}')
        _play_hls(m3u8)

    except Exception as e:
        import traceback
        log(f"[PlayStreamDirect] Exception: {e}\n{traceback.format_exc()}")


def _play_hls_with_headers(hls_url, headers):
    """Play an HLS stream with inputstream.ffmpegdirect and custom headers."""
    # Build URL with pipe headers for ffmpegdirect
    header_str = '&'.join(f'{k}={quote_plus(v)}' for k, v in headers.items())
    full_url = f'{hls_url}|{header_str}'
    
    log(f'[_play_hls_with_headers] Full URL: {full_url[:100]}...')
    
    # Use empty label to preserve the channel name from the list
    liz = xbmcgui.ListItem('', path=full_url)
    liz.setProperty('inputstream', 'inputstream.ffmpegdirect')
    liz.setMimeType('application/x-mpegURL')
    liz.setContentLookup(False)
    
    # Set ffmpegdirect properties
    liz.setProperty('inputstream.ffmpegdirect.is_realtime_stream', 'true')
    liz.setProperty('inputstream.ffmpegdirect.stream_mode', 'timeshift')
    liz.setProperty('inputstream.ffmpegdirect.manifest_type', 'hls')
    
    # Set stream headers property for ffmpegdirect
    headers_formatted = '&'.join(f'{k}={v}' for k, v in headers.items())
    liz.setProperty('inputstream.ffmpegdirect.stream_headers', headers_formatted)
    
    # Also set default headers
    liz.setProperty('inputstream.ffmpegdirect.default_url', hls_url)
    liz.setProperty('inputstream.ffmpegdirect.open_mode', 'curl')
    
    xbmcplugin.setResolvedUrl(addon_handle, True, liz)


def _play_hls(m3u8_url):
    """
    Play an HLS stream using setResolvedUrl.
    
    For AES-128 encrypted HLS from our proxy, native Kodi player works best.
    """
    log(f'[_play_hls] Playing URL: {m3u8_url}')
    
    # Create ListItem for playback
    # Use empty label to preserve the channel name from the list
    liz = xbmcgui.ListItem('', path=m3u8_url)
    liz.setPath(m3u8_url)
    
    # Check user preference from settings (enum: 0=Auto/Native, 1=FFmpegDirect, 2=Adaptive)
    player_mode_idx = addon.getSetting('player_mode') or '0'
    player_modes = ['native', 'ffmpegdirect', 'adaptive']
    try:
        player_mode = player_modes[int(player_mode_idx)]
    except:
        player_mode = 'native'
    log(f'[_play_hls] Player mode: {player_mode}')
    
    if player_mode == 'ffmpegdirect' and xbmc.getCondVisibility('System.HasAddon(inputstream.ffmpegdirect)'):
        log('[_play_hls] Using inputstream.ffmpegdirect')
        liz.setProperty('inputstream', 'inputstream.ffmpegdirect')
        liz.setProperty('inputstream.ffmpegdirect.is_realtime_stream', 'true')
        liz.setProperty('inputstream.ffmpegdirect.stream_mode', 'timeshift')
        liz.setProperty('inputstream.ffmpegdirect.manifest_type', 'hls')
        liz.setProperty('inputstream.ffmpegdirect.open_mode', 'ffmpeg')
    elif player_mode == 'adaptive' and xbmc.getCondVisibility('System.HasAddon(inputstream.adaptive)'):
        log('[_play_hls] Using inputstream.adaptive')
        liz.setProperty('inputstream', 'inputstream.adaptive')
        liz.setProperty('inputstream.adaptive.manifest_type', 'hls')
    else:
        # Native mode: let Kodi's built-in player handle it (no inputstream)
        log('[_play_hls] Using native Kodi player (no inputstream addon)')
    
    log('[_play_hls] Calling setResolvedUrl')
    xbmcplugin.setResolvedUrl(addon_handle, True, liz)


# =============================================================================
# Search Functions
# =============================================================================

def Search_Events():
    keyboard = xbmcgui.Dialog().input("Enter search term", type=xbmcgui.INPUT_ALPHANUM)
    if not keyboard or keyboard.strip() == '':
        return
    term = keyboard.lower().strip()

    try:
        base = get_active_base()
        headers = {'User-Agent': UA, 'Referer': base}
        html_text = requests.get(abs_url('index.php'), headers=headers, timeout=10).text

        events = re.findall(
            r"<div\s+class=\"schedule__event\">.*?"
            r"<div\s+class=\"schedule__eventHeader\"[^>]*?>\s*"
            r"(?:<[^>]+>)*?"
            r"<span\s+class=\"schedule__time\"[^>]*data-time=\"([^\"]+)\"[^>]*>.*?</span>\s*"
            r"<span\s+class=\"schedule__eventTitle\">\s*([^<]+)\s*</span>.*?"
            r"</div>\s*"
            r"<div\s+class=\"schedule__channels\">(.*?)</div>",
            html_text, re.IGNORECASE | re.DOTALL
        )

        rows = {}
        seen = set()
        for time_str, raw_title, channels_block in events:
            title_clean = html.unescape(raw_title.strip())
            if term not in title_clean.lower():
                continue

            local_ts = get_local_time(time_str.strip())
            row_title = f"[COLOR gold]{local_ts}[/COLOR] {title_clean}"

            for href, title_attr, link_text in re.findall(
                r"<a[^>]+href=\"([^\"]+)\"[^>]*title=\"([^\"]*)\"[^>]*>(.*?)</a>",
                channels_block, re.IGNORECASE | re.DOTALL
            ):
                try:
                    u = urlparse(href)
                    qs = dict(parse_qsl(u.query))
                    cid = (qs.get('id') or '').strip()
                except Exception:
                    cid = ''
                if not cid:
                    continue

                sig = (title_clean.lower(), time_str.strip(), cid)
                if sig in seen:
                    continue
                seen.add(sig)

                ch_name = html.unescape((title_attr or link_text or '').strip()) or 'Channel'
                rows.setdefault(row_title, [])
                if all(cid != c['channel_id'] for c in rows[row_title]):
                    rows[row_title].append({'channel_name': ch_name, 'channel_id': cid})

        if not rows:
            xbmcgui.Dialog().ok("Search", "No matching events found.")
            return

        for row_title, chans in rows.items():
            addDir(row_title, build_url({'mode': 'trList', 'trType': 'search', 'channels': json.dumps(chans)}), True)
        closeDir()

    except Exception as e:
        log(f"Search_Events error: {e}")
        xbmcgui.Dialog().ok("Search", "Search failed. Check log.")


def Search_Channels():
    keyboard = xbmcgui.Dialog().input("Enter channel name", type=xbmcgui.INPUT_ALPHANUM)
    if not keyboard or keyboard.strip() == '':
        return
    term = keyboard.lower().strip()

    try:
        base = get_active_base()
        headers = {'User-Agent': UA, 'Referer': base}
        html_text = requests.get(abs_url('index.php'), headers=headers, timeout=10).text

        channel_blocks = re.findall(
            r"<div\s+class=\"schedule__event\">.*?"
            r"<div\s+class=\"schedule__eventHeader\"[^>]*?>.*?</div>\s*"
            r"<div\s+class=\"schedule__channels\">(.*?)</div>",
            html_text, re.IGNORECASE | re.DOTALL
        )

        results = []
        seen = set()
        for block in channel_blocks:
            for href, title_attr, link_text in re.findall(
                r"<a[^>]+href=\"([^\"]+)\"[^>]*title=\"([^\"]*)\"[^>]*>(.*?)</a>",
                block, re.IGNORECASE | re.DOTALL
            ):
                display = html.unescape((title_attr or link_text or '').strip())
                if term not in display.lower():
                    continue

                try:
                    u = urlparse(href)
                    qs = dict(parse_qsl(u.query))
                    cid = (qs.get('id') or '').strip()
                except Exception:
                    cid = ''
                if not cid:
                    continue

                sig = (display.lower(), cid)
                if sig in seen:
                    continue
                seen.add(sig)

                results.append({'title': display, 'channel_id': cid})

        if not results:
            xbmcgui.Dialog().ok("Search", "No matching channels found.")
            return

        for result in results:
            addDir(result['title'], build_url({
                'mode': 'trLinks',
                'trData': json.dumps({'channels': [{'channel_name': result["title"], 'channel_id': result["channel_id"]}]})
            }), False)
        closeDir()

    except Exception as e:
        log(f"Search_Channels error: {e}")
        xbmcgui.Dialog().ok("Search", "Search failed. Check log.")


# =============================================================================
# Settings Actions
# =============================================================================

def refresh_active_base():
    new_base = resolve_active_baseurl(SEED_BASEURL)
    set_active_base(new_base)
    xbmcgui.Dialog().ok(ADDON_NAME, f"Active domain set to:\n{new_base}")
    xbmc.executebuiltin('Container.Refresh')


# =============================================================================
# Main Entry Point
# =============================================================================

kodiversion = getKodiversion()
mode = params.get('mode', None)

if not mode:
    Main_Menu()
else:
    if mode == 'menu':
        servType = params.get('serv_type')
        if servType == 'sched':
            Menu_Trans()
        elif servType == 'live_tv':
            list_gen()
        elif servType == 'search':
            Search_Events()
        elif servType == 'search_channels':
            Search_Channels()
        elif servType == 'refresh_sched':
            xbmc.executebuiltin('Container.Refresh')
        elif servType == 'resolve_base_now':
            refresh_active_base()
        elif servType == 'backend_status':
            show_backend_status()

    elif mode == 'showChannels':
        transType = params.get('trType')
        channels = getTransData(transType)
        ShowChannels(transType, channels)

    elif mode == 'trList':
        transType = params.get('trType')
        channels = json.loads(params.get('channels'))
        TransList(transType, channels)

    elif mode == 'trLinks':
        trData = params.get('trData')
        getSource(trData)

    elif mode == 'play':
        link = params.get('url')
        PlayStream(link)

    elif mode == 'resolve_base_now':
        refresh_active_base()
